/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: timer.cc,v 1.4 2002/09/02 18:38:16 jsalz Exp $
 *
 * A class enabling handlers to receive timed events.
 *
 */

#include "config.h"

#include "tesla/timer.hh"
#include "tesla/tesla.h"

#include <sys/time.h>
#include <set>
#include <vector>

const long long timer::CLEAR = 0x7FFFFFFFFFFFFFFFLL;

using namespace std;

struct lttimerref {
    bool operator()(timer *a, timer *b) const {
	return a->time() < b->time();
    }
};

typedef set<timer*, lttimerref> timerset;
static timerset* all;

long long timer::now() {
    timeval tv;
    gettimeofday(&tv, 0);
    return tv.tv_sec * 1000000LL + tv.tv_usec;
}

timer::timer(long long when, from which_base, string desc) : when(CLEAR) {
    if (!all)
	all = new timerset;

    arm(when, which_base);
}

timer::~timer() {
    arm(CLEAR);
}

void timer::arm(long long when, from which_base) {
    if (when != CLEAR && which_base == FROM_NOW)
	when += now();

    if (this->when == when)
	return;

    if (this->when != CLEAR)
	all->erase(this);
    this->when = when;
    if (this->when != CLEAR)
	all->insert(this);
}

long long timer::fire_all() {
    if (!all) return CLEAR;

    static vector<timer*> to_fire;
    long long start = now();
    timerset::iterator i;

    to_fire.clear();

    for (i = all->begin(); i != all->end(); ++i) {
	if ((*i)->when <= start)
	    (*i)->when = CLEAR;
	else
	    break;
    }

    to_fire.insert(to_fire.begin(), all->begin(), i);
    all->erase(all->begin(), i);

    for (vector<timer*>::iterator j = to_fire.begin(); j != to_fire.end(); ++j)
	(*j)->fire();

    if (all->empty())
	return CLEAR;
    else {
	long long time = (*(all->begin()))->when - timer::now();
	if (time > 0LL)
	    return time;
	else
	    return 0LL;
    }
}
