/*
 * TESLA
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: top_handler.cc,v 1.14 2002/10/04 03:38:37 jsalz Exp $
 *
 * The top handler, which handles communication between the
 * application and the master process.
 *
 */

#include "config.h"

#include <set>

#include "tesla/tesla.h"
#include "tesla-internal.h"
#include "top_handler.hh"
#include "teslamaster.hh"

extern set<top_handler*> close_set;

top_handler::top_handler(int _fd, int _type, int _conn_id, flow_handler *_h) :
    async(_fd), flow_handler(init_context(_info, *(flow_handler*)0, -1, -1, -1), false),
    fd(_fd), conn_id(_conn_id), h(_h), type(_type),
    shut_read(false), shut_write(false),
    is_connected(false),
    closed(false)
{
    if (h) h->set_upstream(this);

    if (fd >= 0)
	set_ractive(true);
}

top_handler::~top_handler()
{
    ts_debug_1("Deleting top_handler (fd=%d)", fd);
    close();
    if (h)
	delete h;
}

// If we've been shut down in both directions, remember to close
// this handler off later
void top_handler::check_shutdown()
{
    if (!closed && (shut_read || !is_connected) && shut_write) {
	//    if (!closed && shut_read && shut_write) {
	ts_debug_1("Marking %d for later closing", fd);
	close_set.insert(this);
    }
}

int top_handler::close() {
    if (closed)
	return -1;

    closed = true;

    // Close connection to app
    ::close(fd);

    // Tell handler to close
    if (h)
	return h->close();

    return 0;
}

// Called when we get an EPIPE error writing to the application
void top_handler::sigpipe() {
    // No more bytes to write to application
	
    // Shut down the network stack; never read from stack again
    shutdown(true, false);

    shut_read = true;
    check_shutdown();
}

// Called when bytes are available from application socket
void top_handler::ravail() {
    static ts_dgram_header dg;
    static char buf[MAX_DGRAM_SIZE];
    size_t bytes;

    if (type == SOCK_DGRAM) {
	// For datagrams, read header and then data
	bytes = read_fully(fd, &dg, sizeof dg);
	ts_debug_1("Got datagram on %d, payload length %d", fd, dg.length);
	if (bytes > 0) {
	    assert(dg.length <= sizeof buf);
	    bytes = read_fully(fd, buf, dg.length);
	}
    } else {
	// For non-datagrams, read up to DATA_BUFFER_SIZE bytes at a
	// time
	bytes = ::read(fd, buf, DATA_BUFFER_SIZE);
    }

    if (bytes <= 0) {
	ts_debug_1("Application has done shutdown(write) on %d", fd);

	// Don't bother reading any more bytes
	set_ractive(false);

	// Tell handlers they'll never be getting any more bytes
	h->shutdown(false, true);

	shut_write = true;
	check_shutdown();
	return;
    }

    int ret;
    if (type == SOCK_DGRAM) {
      ret = h->write(data(buf, dg.length, 0,
			  &dg.addr, dg.addrlen));
    } else {
	// Non-datagram: just throw the bytes down the chain
	ts_debug_2("Got bytes to write to flow_handler");
	ret = h->write(data(buf, bytes));
    }

    if (ret < 0) {
	ts_debug_1("Got error %s writing to stack", strerror(-ret));

	// Shut down the application
	set_ractive(false);
	::shutdown(fd, SHUT_RD);
	shut_write = true;
	check_shutdown();
    }
}

void top_handler::wavail() {
    // We can write to the application now.  Presumably there's data
    // buffered in buffer (otherwise we wouldn't have called
    // set_wactive(true)).

    const char *p = buffer.data();
    int len = buffer.length();

    while (len > 0) {
	// Write as much as we can
	int written = ::write(fd, p, len);

	if (written == -1 && errno == EAGAIN) {
	    // Can't write any more bytes; buffer the rest.  This
	    // is pretty inefficient
	    buffer = string(buffer, len, buffer.length());
	    return;
	}
	if (written == -1 && errno == EPIPE) {
	    // The other end of the pipe has closed (app has done
	    // shutdown(fd, SHUT_READ))
	    sigpipe();
	    return;
	}

	if (written <= 0) {
	    // Buffer whatever's left and bail.
	    ts_debug_1("Buffering data...");
	    buffer = string(buffer, len, buffer.length());
	    set_wactive(false);
	    set_ractive(true);
	    return;
	}

	p += written;
	len -= written;
    }

    // We're done (len == 0).  Unthrottle downstream handlers,
    // and stop waiting for write availability to application.
    h->may_avail(true);
    set_wactive(false);
    set_ractive(true);
    buffer.resize(0);
}

void top_handler::connected(flow_handler *from, bool success) {
    // Write "B" or "A" to the app to indicate connection success
    // or failure
    if (type != SOCK_DGRAM)
	avail(from, success ? "B" : "A");

    if (success)
	is_connected = true;
}

void top_handler::accept_ready(flow_handler *from) {
    avail(from, "C");
}

string top_handler::getsockopt(int level, int optname, int maxlen) {
    if (!h) return string();
    return h->getsockopt(level, optname, maxlen);
}
int top_handler::setsockopt(int level, int optname, string value) {
    if (!h) return -1;
    return h->setsockopt(level, optname, value);
}
int top_handler::ioctl(string target, int optname, string value, string& out) {
    if (!h) return -EINVAL;
    return h->ioctl(target, optname, value, out);
}
address top_handler::getpeername() {
    if (!h) return address();
    return h->getpeername();
}
address top_handler::getsockname() {
    if (!h) return address();
    return h->getsockname();
}
    
// Need to write bytes to application.
bool top_handler::avail(flow_handler *from, data d) {
    if (d.length() == 0) {
	// No more bytes from stack
	ts_debug_1("Network has shutdown its writes on %d (errno=%d)", fd, d.error());

	if (!shut_read) {
	    // Never read from stack again
	    shut_read = true;
	    // Never write to the application again
	    ::shutdown(fd, SHUT_WR);
	    check_shutdown();
	}

	return false;
    }

    if (type == SOCK_DGRAM) {
	ts_debug_2("avail sock_dgram");

	ts_dgram_header dg;

	dg.length = d.length();
	dg.addrlen = min(sizeof dg.addr, d.addrlen());
	if (d.addr())
	    memcpy(&dg.addr, d.addr(), dg.addrlen);

	return
	    sendapp_or_buffer(&dg, sizeof dg) &&
	    sendapp_or_buffer(d.bits(), d.length());
    }

    return sendapp_or_buffer(d.bits(), d.length());
}

bool top_handler::sendapp_or_buffer(const void *pp, int len) {
    const char *p = (const char *)pp;

    if (buffer.size() != 0) {
	buffer.append(p, len);
	return true;
    }

    while (len > 0) {
	int written = ::write(fd, p, len);

	ts_debug_2("- system avail call on %d returned %d (out of %d)", fd, written, len);

	if (written == -1 && errno == EAGAIN) {
	    ts_debug_2("  - buffering remaining %d bytes", len);

	    // buffer the rest, select on write, and throttle downstream
	    buffer.append(p, len);
	    set_wactive(true);
	    h->may_avail(false);
	    return true;
	}

	if (written <= 0) {
	    ts_debug_2("  - closed for writing");

	    // closed for writing
	    return false;
	}

	ts_debug_2("  - wrote %d bytes successfully", written);
	
	p += written;
	len -= written;
    }

    return true;
}

void top_handler::may_write(flow_handler *from, bool may) {
    // If we get a may_write(false), don't bother reading from the
    // application.
    can_write = may;
    set_ractive(may);
}

bool top_handler::may_exit() {
    if (!h) return true;
    return h->may_exit();
}

flow_handler::info top_handler::_info("top_handler", "top_handler", "", -1, -1, 0);
