/*
 * TESLA: A Transparent, Extensible Session-Layer Architecture
 *
 * Jon Salz <jsalz@mit.edu>
 * Alex C. Snoeren <snoeren@lcs.mit.edu>
 *
 * Copyright (c) 2001-2 Massachusetts Institute of Technology.
 *
 * This software is being provided by the copyright holders under the GNU
 * General Public License, either version 2 or, at your discretion, any later
 * version. For more information, see the `COPYING' file in the source
 * distribution.
 *
 * $Id: utility.c,v 1.14 2002/10/10 17:24:41 snoeren Exp $
 *
 * Utility functions for libtesla.so.  #included by tesla.c.
 *
 */

// Read count bytes from a file descriptor (blocking).
// Return either count (success) or 0 (failure).
static int read_fully(int fd, void *buf, size_t count)
{
    int orig_count = count;
    while (count) {
        int bytes = TS_CALL_LIBC(read, fd, buf, count);
        if (bytes <= 0)
            return 0;
        assert(bytes <= count);
        buf += bytes;
        count -= bytes;
    }
    return orig_count;
}

// Write count bytes to a file descriptor (blocking).
// Return either count (success) or 0 (failure).
static int write_fully(int fd, const void *buf, size_t count)
{
    int orig_count = count;
    while (count) {
        int written = TS_CALL_LIBC(write, fd, buf, count);
        if (written <= 0)
            return 0;
        assert(written <= count);
        buf += written;
        count -= written;
    }
    return orig_count;
}

// Send a message to the master.  The msg->id field is set automatically.
static unsigned int send_master(master_msg_t *msg) {
    msg->id = next_msg_id++;
    msg->pid = getpid();
    msg->len = sizeof *msg - sizeof msg->len;

    ts_debug_2("Sending message to master: id=%d, pid=%d, len=%d, type=%s",
	       msg->id, msg->pid, msg->len, MSG_TYPE(msg->type));

    if (write_fully(master_fd, msg, sizeof *msg) != sizeof *msg)
	ts_fatal("Unable to send message to master: %s", strerror(errno));

    return msg->id;
}

static unsigned int send_master_header(master_msg_t *msg, int len) {
    msg->len = MASTER_MSG_HDRLEN + len - sizeof msg->len;
    msg->id = next_msg_id++;
    msg->pid = getpid();

    ts_debug_2("Sending partial message to master: id=%d, pid=%d, len=%d, type=%s",
	       msg->id, msg->pid, msg->len, MSG_TYPE(msg->type));

    if (write_fully(master_fd, msg, MASTER_MSG_HDRLEN) != MASTER_MSG_HDRLEN)
    {
	ts_fatal("Unable to send message to master: %s", strerror(errno));
    }

    return msg->id;
}

static unsigned int recv_master(master_msg_t *msg) {
    int bytes;

    ts_debug_2("Reading length from master");
    if ((bytes = read_fully(master_fd, &msg->len, sizeof msg->len)) != sizeof msg->len)
	ts_fatal("Expected length from master but got %d bytes", bytes);
    ts_debug_2("Reading data (%d bytes) from master", msg->len);
    if ((bytes = read_fully(master_fd, &msg->id, msg->len)) != msg->len)
	ts_fatal("Expected %d bytes from master but got %d bytes", msg->len, bytes);

    ts_debug_2("Received message: id=%d, length=%d, type=%s", msg->id, msg->len, MSG_TYPE(msg->type));
    if (msg->type == MSG_FH_FOLLOWS) {
	ts_debug_2(" - waiting for FD (FH id was %d)", msg->conn_id);
	msg->body.fh_pass.fh = get_fd_from_master();
	ts_debug_2(" - got FD");
    }

    return msg->id;
}

static void cleanup(void)
{
    master_msg_t msg;

    ts_debug_2("In cleanup");
    if (master_fd >= 0) {
	msg.type = MSG_NAK;
	send_master(&msg);
    }
}

static void init()
{
    const char *uidrealstr;
    const char *oldstatefile;

    __ts_libc = 0;
    is_enabled = 0;

    uidrealstr = getenv("TS_SETUID");
    if (uidrealstr) {
	uid_t uidreal = atoi(uidrealstr);

	__ts_disable_debug = 1;
	is_suid = 1;
	require_uid = uidreal;

	if (setuid(uidreal)) {
	    fprintf(stderr, "Unable to setuid to %d: %s", uidreal, strerror(errno));
	    exit(1);
	}
    }
    if((oldstatefile = getenv("TS_RESTORE_FILE"))) {
      ts_restore_state(oldstatefile);
      unlink(oldstatefile);
    }
}

// Called only by tesla_enabled(1), or ts_restore_state
static void enable(int state_fd)
{
    master_msg_t msg;
    int sv[2];
    unsigned int id;
    pid_t parent_pid = getpid();
    pid_t pid;

    if (socketpair(AF_UNIX, SOCK_STREAM, 0, sv))
	ts_fatal("Unable to create socketpair: %s", strerror(errno));

    if ((pid = fork()) == 0) {
	/* Master process.  Pass control on to teslamaster */
	char pidstr[32];
	char fdstr[32];
	char statefdstr[32];

	close(sv[0]);

	sprintf(pidstr, "%d", parent_pid);
	sprintf(fdstr, "%d", sv[1]);
	sprintf(statefdstr, "%d", state_fd);

	unsetenv("LD_PRELOAD");
	
	execlp(TESLA_MASTER_PATH, TESLA_MASTER_PATH, pidstr, fdstr, statefdstr, 0);

	ts_fatal("Unable to exec teslamaster: %s", strerror(errno));
    }

    close(sv[1]);
    master_fd = sv[0];

    // Wait for master
    wait(0);

    /* Child process. */
    msg.type = MSG_HELLO;
    id = send_master(&msg);

    if (recv_master(&msg) != id || msg.type != MSG_HELLO)
	ts_fatal("Did not receive HELLO from master, %d, %d", id, msg.type);

    atexit(cleanup);
}

#define	CONTROLLEN	(sizeof(struct cmsghdr) + sizeof(int))
static int get_fd_from(int src_fd) {
    /* From Perl File::FDpasser */
    static struct cmsghdr *cmptr = 0;

    int newfd = -1, nread, status;
    char *ptr, buf[4096];
    struct iovec iov[1];
    struct msghdr msg;
  
    status = -1;
    iov[0].iov_base = buf;
    iov[0].iov_len  = sizeof(buf);
    msg.msg_iov     = iov;
    msg.msg_iovlen  = 1;
    msg.msg_name    = NULL;
    msg.msg_namelen = 0;
    if (cmptr == NULL && (cmptr = malloc(CONTROLLEN)) == NULL) return(-1);
    msg.msg_control    = (caddr_t) cmptr;
    msg.msg_controllen = CONTROLLEN;
    
    if ( (nread = recvmsg(src_fd, &msg, 0)) < 0) return(-1);
    else if (nread == 0)
	ts_fatal("Connection closed by master while reading filehandle");

    for (ptr = buf; ptr < &buf[nread]; ) {
	if (*ptr++ == 0) {
	    if (ptr != &buf[nread-1]) { ts_fatal("message format error"); }
	    status = *ptr & 255;
	    if (status == 0) {
		if (msg.msg_controllen != CONTROLLEN) { ts_fatal("status = 0 but no fd"); }
		newfd = *(int *)CMSG_DATA(cmptr); /* new descriptor */
	    } else newfd = -status;
	    nread -= 2;
	}
    }
    if (nread > 0) if (write(STDERR_FILENO, buf, nread) != nread) { ts_fatal("???"); }
    if (status >= 0) return(newfd);       /* final data has arrived,  descriptor, or -status */

    return 0;
}

static int get_fd_from_master() {
    return get_fd_from(master_fd);
}

static int set_message_handler(ts_message_handler handler, int sig)
{
    struct sigaction act;
    master_msg_t msg;
    int id;

    ts_debug_1("Setting message handler to %p, using signal %d", handler, sig);

    if (handler == 0) {
	if (msg_handler) {
	    ts_debug_2("Telling master to disable async messages");
	    msg.type = MSG_ASYNC_MESSAGES;
	    msg.body.async_messages.enabled = 0;
	    id = send_master(&msg);
	    if (recv_master(&msg) != id)
		ts_fatal("Mismatched message");
	    if (msg.type != MSG_ACK)
		ts_fatal("Master unable to disable async messages");

	    ts_debug_2(" - OK!");

	    msg_handler = 0;
	    signal(msg_signal, SIG_IGN);
	    msg_signal = 0;
	}
	return 0;
    }

    assert(sig != 0);
    
    msg_handler = handler;
    msg_signal = sig;

    memset(&act, 0, sizeof act);
    act.sa_sigaction = msg_sigaction;
    act.sa_flags = SA_RESTART | SA_SIGINFO;
    sigaction(sig, &act, 0);

    ts_debug_2("Telling master to enable async messages");
    msg.type = MSG_ASYNC_MESSAGES;
    msg.body.async_messages.enabled = 1;
    msg.body.async_messages.pass = msg_fd == -1;
    id = send_master(&msg);
    if (recv_master(&msg) != id)
	ts_fatal("Mismatched message");

    if (msg_fd == -1) {
	if (msg.type != MSG_FH_FOLLOWS)
	    ts_fatal("Master did not pass async message FD");
	msg_fd = msg.body.fh_pass.fh;
    } else if (msg.type != MSG_ACK) {
	ts_fatal("Unable to enable async messages");
    }

    ts_debug_2(" - Async messages enabled! msg_fd is %d", msg_fd);
    fcntl(msg_fd, F_SETOWN, getpid());
#ifdef F_SETSIG
    fcntl(msg_fd, F_SETSIG, sig);
#else
    if (sig != SIGIO)
        ts_error("Warning: F_SETSIG not available; using SIGIO for async messages");
#endif
    fcntl(msg_fd, F_SETFL, O_ASYNC);

    return 0;
}

static void msg_sigaction(int signal, siginfo_t *info, void *arg)
{
    fd_set fds;
    struct timeval tv;
    ts_async_message_s msg;
    char *buf;
    int recv_fd;

    /* Bail if nothing ready to be read */
    FD_ZERO(&fds);

    while (1) {
	FD_SET(msg_fd, &fds);
	tv.tv_sec = 0;
	tv.tv_usec = 0;
	if (select(msg_fd+1, &fds, 0, 0, &tv) == 0)
	    break;

	if (!read_fully(msg_fd, &msg, sizeof msg)) {
	    ts_error("Unable to read async message header");
	    return;
	}

	ts_debug_1("Async message; conn_id %d; handler %s; name %d; data_length %d",
		   msg.conn_id, msg.handler, msg.name, msg.data_length);

	if (msg.data_length != -1) {
	    buf = malloc(msg.data_length + 1);
	    read_fully(msg_fd, buf, msg.data_length);

	    /* Null-terminate */
	    buf[msg.data_length] = 0;
	} else {
	    recv_fd = get_fd_from(msg_fd);
	    buf = 0;
	    msg.data_length = recv_fd; // XXX: Hack!!!
	}
	    
	if (msg_handler) {
	    int fd = -1;
	    int i;
	    
	    // Figure out which FD it refers to
	    for (i = 0; i < FD_SETSIZE; ++i) {
		if (ts_fds[i].wrapped && msg.conn_id == ts_fds[i].conn_id) {
		    fd = i;
		    break;
		}
	    }

	    (*msg_handler)(fd, msg.handler, msg.name, buf, msg.data_length);
	}
	free(buf);
    }
}

static int tesla_enabled(int enable_if_not)
{
    if (is_enabled)
	return 1;
    if (!enable_if_not)
	return 0;

    // TESLA is not enabled, but we want to enable it

    if (is_suid) {
	if (getuid() != require_uid || geteuid() != require_uid)
	    return 0;

	// OK, SUID has been dropped.  Just to be sure, and to kill
	// saved set-user-id...
#if 0
	if (setresuid(require_uid, require_uid, require_uid)) {
	    fprintf(stderr, "Unable to setresuid(%d,%d,%d)", require_uid, require_uid, require_uid);
	    exit(1);
	}
#endif

	is_suid = 0;
    }

    // Enable code goes here
    is_enabled = 1;
    __ts_disable_debug = 0;
    enable(-1);
    ts_debug_1("TESLA enabled");

    return 1;
}
